/*
 * Copyright (c) 2003-2010, John Wiegley.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * - Neither the name of New Artisans LLC nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @addtogroup util
 *
 * @file   system.hh
 * @author John Wiegley
 *
 * @brief  All system headers needed by Ledger.
 *
 * These are collected here so that a pre-compiled header can be made.
 * None of these header files (with the exception of acconf.h, when
 * configure is re-run) are expected to change.
 */
#include "config.h"

#if defined(__GNUG__) && __GNUG__ < 3
#define _XOPEN_SOURCE
#endif

#include <algorithm>
#include <exception>
#include <typeinfo>
#include <locale>
#include <stdexcept>
#include <iostream>
#include <streambuf>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <iterator>
#include <list>
#include <map>
#include <memory>
#include <new>
#include <set>
#include <stack>
#include <string>
#include <vector>

#if defined(__GNUG__) && __GNUG__ < 3

namespace std {
  inline ostream & right (ostream & i) {
    i.setf(i.right, i.adjustfield);
    return i;
  }
  inline ostream & left (ostream & i) {
    i.setf(i.left, i.adjustfield);
    return i;
  }
}

typedef std::streamoff istream_pos_type;
typedef std::streamoff ostream_pos_type;

#else // ! (defined(__GNUG__) && __GNUG__ < 3)

typedef std::istream::pos_type istream_pos_type;
typedef std::ostream::pos_type ostream_pos_type;

#endif

#include <cassert>
#include <cctype>
#include <cstdarg>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <csignal>

#if defined __FreeBSD__ && __FreeBSD__ <= 4
// FreeBSD has a broken isspace macro, so don't use it
#undef isspace(c)
#endif

#include <sys/stat.h>
#if defined(WIN32)
#include <io.h>
#else
#include <unistd.h>
#endif
#if defined(HAVE_GETPWUID) || defined(HAVE_GETPWNAM)
#include <pwd.h>
#endif

#if defined(HAVE_UNIX_PIPES)
#include <sys/types.h>
#include <sys/wait.h>
#endif
#if defined(HAVE_GETTEXT)
#include "gettext.h"
#define _(str) gettext(str)
#else
#define _(str) str
#endif

#include <gmp.h>
#include <mpfr.h>
#include "sha1.h"
#include "utf8.h"

#if defined(HAVE_LIBEDIT)
#include <editline/readline.h>
#endif

#include <boost/algorithm/string.hpp>
#include <boost/any.hpp>
#include <boost/bind.hpp>
#include <boost/cast.hpp>
#include <boost/current_function.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/date_time/posix_time/posix_time_io.hpp>
#include <boost/date_time/gregorian/gregorian_io.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/exception.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#if !(defined(__GXX_EXPERIMENTAL_CXX0X__) && __GXX_EXPERIMENTAL_CXX0X__)
#include <boost/foreach.hpp>
#endif
#include <boost/function.hpp>
#include <boost/intrusive_ptr.hpp>
#include <boost/iostreams/stream.hpp>
#include <boost/iostreams/write.hpp>
#define BOOST_IOSTREAMS_USE_DEPRECATED 1
#include <boost/iostreams/device/file_descriptor.hpp>
#include <boost/iterator/iterator_facade.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/operators.hpp>
#include <boost/optional.hpp>
#include <boost/ptr_container/ptr_list.hpp>
#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_int.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/random/variate_generator.hpp>
#if defined(HAVE_BOOST_REGEX_UNICODE)
#include <boost/regex/icu.hpp>
#else
#include <boost/regex.hpp>
#endif // HAVE_BOOST_REGEX_UNICODE
#include <boost/variant.hpp>
#include <boost/version.hpp>

#if defined(HAVE_BOOST_SERIALIZATION)

#include <boost/archive/binary_iarchive.hpp>
#include <boost/archive/binary_oarchive.hpp>

#include <boost/serialization/base_object.hpp>
#include <boost/serialization/binary_object.hpp>
#include <boost/serialization/optional.hpp>
#include <boost/serialization/shared_ptr.hpp>
#include <boost/serialization/variant.hpp>
#include <boost/serialization/utility.hpp>
#include <boost/serialization/export.hpp>
#include <boost/serialization/level.hpp>
#include <boost/serialization/string.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/serialization/deque.hpp>
#include <boost/serialization/list.hpp>
#include <boost/serialization/map.hpp>

#include <boost/date_time/posix_time/time_serialize.hpp>
#include <boost/date_time/gregorian/greg_serialize.hpp>
#include <boost/ptr_container/serialize_ptr_deque.hpp>

namespace boost {
namespace serialization {

template <class Archive>
void serialize(Archive& ar, boost::filesystem::path& p, const unsigned int)
{
  std::string s;
  if (Archive::is_saving::value)
    s = p.string();

  ar & s;

  if (Archive::is_loading::value)
    p = s;
}

template <class Archive, class T>
void serialize(Archive& ar, boost::intrusive_ptr<T>& ptr, const unsigned int)
{
  if (Archive::is_saving::value) {
    T * p = ptr.get();
    ar & p;
  }
  else if (Archive::is_loading::value) {
    T * p;
    ar & p;
    ptr.reset(p);
  }
}

template <class Archive, class T>
void serialize(Archive&, boost::function<T>&, const unsigned int)
{
}

template <class Archive>
void serialize(Archive& ar, istream_pos_type& pos, const unsigned int)
{
  ar & make_binary_object(&pos, sizeof(istream_pos_type));
}

} // namespace serialization
} // namespace boost

#else // HAVE_BOOST_SERIALIZATION

#include <boost/ptr_container/ptr_deque.hpp>

#endif // HAVE_BOOST_SERIALIZATION

#if defined(HAVE_BOOST_PYTHON)

#include <boost/python.hpp>

#include <boost/python/detail/wrap_python.hpp>
#include <datetime.h>
#include <unicodeobject.h>

#include <boost/python/module_init.hpp>
#include <boost/python/suite/indexing/vector_indexing_suite.hpp>

#endif // HAVE_BOOST_PYTHON
